# ---------------------------------------------------------------------------------------------------------------------
# PARAMETRY WYMAGANE
# Konieczne jest podanie wartości dla każdego z tych parametrów.
# ---------------------------------------------------------------------------------------------------------------------

variable "cluster_name" {
  description = "Nazwa używana we wszystkich zasobach klastra"
  type        = string
}

variable "ami" {
  description = "Obraz AMI do uruchomienia w klastrze"
  type        = string
}

variable "instance_type" {
  description = "Typ egzemplarza EC2 do uruchomienia (np. t2.micro)"
  type        = string

  # Przykład wymuszenia stosowania ograniczeń biznesowych za pomocą bloku validation.
  # Tutaj ograniczamy typ egzemplarza do jednego z dostępnych w ramach bezpłatnego konta AWS.
  validation {
    condition     = contains(["t2.micro", "t3.micro"], var.instance_type)
    error_message = "Konto bezpłatne pozwala na użycie jedynie: t2.micro | t3.micro."
  }
}

variable "min_size" {
  description = "Minimalna liczba egzemplarzy EC2 w ASG"
  type        = number

  validation {
    condition     = var.min_size > 0
    error_message = "Grupa ASG nie może być pusta, ponieważ wtedy mamy awarię!"
  }

  validation {
    condition     = var.min_size <= 10
    error_message = "Grupa ASG powinna mieć co najwyżej 10 egzemplarzy, aby ograniczać koszty."
  }
}

variable "max_size" {
  description = "Maksymalna liczba egzemplarzy EC2 w ASG"
  type        = number

  validation {
    condition     = var.max_size > 0
    error_message = "Grupa ASG nie może być pusta, ponieważ wtedy mamy awarię!"
  }

  validation {
    condition     = var.max_size <= 10
    error_message = "Grupa ASG powinna mieć co najwyżej 10 egzemplarzy, aby ograniczać koszty."
  }
}

variable "subnet_ids" {
  description = "Identyfikatory podsieci do wdrożenia"
  type        = list(string)
}

variable "enable_autoscaling" {
  description = "Wartość true oznacza włączenie automatycznego skalowania"
  type        = bool
}

# ---------------------------------------------------------------------------------------------------------------------
# PARAMETRY OPCJONALNE
# Te parametry mają rozsądne wartości domyślne.
# ---------------------------------------------------------------------------------------------------------------------

variable "target_group_arns" {
  description = "Grupy docelowe wartości ARN w mechanizmie ELB, w którym zostaną zarejestrowane egzemplarze"
  type        = list(string)
  default     = []
}

variable "health_check_type" {
  description = "Typ przeprowadzanego sprawdzenia stanu. To musi być EC2 lub ELB."
  type        = string
  default     = "EC2"

  validation {
    condition     = contains(["EC2", "ELB"], var.health_check_type)
    error_message = "Typ przeprowadzanego sprawdzenia stanu. To musi być EC2 lub ELB."
  }
}

variable "user_data" {
  description = "Skrypt danych użytkownika przeznaczony do wykonania w każdym egzemplarzu podczas jego uruchmamiania"
  type        = string
  default     = null
}

variable "custom_tags" {
  description = "Własne tagi przeznaczone do użycia w egzemplarzach ASG"
  type        = map(string)
  default     = {}
}

variable "server_port" {
  description = "Numer portu używany przez serwer dla żądań HTTP"
  type        = number
  default     = 8080
}
